/*
 * StunInfo.h
 *
 * Created 8/24/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/24/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Only CharacterObject type Objects can be stunned. Make sure to supply a 
  * CharacterObject type when creating an ActionTask or EffectTask using
  * a StunInfo.
  */
 
 #ifndef STUN_INFO_H
 #define STUN_INFO_H
 
 template <typename T> class StunInfo;
 
 #include "global.h"
 
 #include "ActionInfo.h"
 #include "pointerTo.h"
 #include "CharacterObject.h"
 
 /**
  * Class specification for StunInfo
  */
 template <typename T>
 class StunInfo : virtual public ActionInfo<T>
 {
 // Static Functions
 private:
    static inline AsyncTask::DoneStatus process_stun( GenericAsyncTask* task_Ptr, void* data_Ptr );
    static inline void upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr );
 
 // Data Members
 private:
    double _stun_duration;
    
 // Local Functions
 public:
    StunInfo( const double stun_duration = 0.0 );
    StunInfo( const StunInfo<T>& stun_info );
    virtual ~StunInfo();
    inline StunInfo<T>& operator=( const StunInfo<T>& stun_info );
    virtual inline void split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr );
 
 // Private Functions
 private:
    
 // Static Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * process_stun() keeps the CharacterObject stunned until the specified duration.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus StunInfo<T>::process_stun( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    nassertr( task_Ptr != NULL, AsyncTask::DS_done );
    
    PT(ActionTask<T>) action_task_Ptr( dynamic_cast<ActionTask<T>*>( task_Ptr ) );
    PT(StunInfo<T>) stun_info_Ptr( dynamic_cast<StunInfo<T>*>(action_task_Ptr->get_action_info()) );
    nassertr( stun_info_Ptr != NULL, AsyncTask::DS_done );
    
    // if ( current_time - action_invoke_time > stun_duration )
    if ( global::_clock_Ptr->get_real_time() - stun_info_Ptr->get_time_action_was_invoked() > stun_info_Ptr->_stun_duration )
    {
        //PT(Object<T>) object_Ptr( action_task_Ptr->get_object() );
        //nassertr( object_Ptr != NULL, AsyncTask::DS_done );
        
        //object_Ptr->unstun();
        
        return AsyncTask::DS_done;
    }
    /*else
    {
        //PT(Object<T>) object_Ptr( action_task_Ptr->get_object() );
        //nassertr( object_Ptr != NULL, AsyncTask::DS_done );
        
        // make character stun if is not already stunned
        //object_Ptr->stun();
    }*/
    
    return AsyncTask::DS_cont;
 }
 
 /**
  * upon_death_handler() decrements the stun counter for the CharacterObject.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (bool) clean_exit
  * @param (void*) data_Ptr
  */
 template <typename T>
 inline void StunInfo<T>::upon_death_handler( GenericAsyncTask* task_Ptr, bool clean_exit, void* data_Ptr )
 {
    nassertv( task_Ptr != NULL );
    
    //if ( !clean_exit ) // if the task did not finish by returning AsyncTask::DS_done
    //{
        PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
        PT(StunInfo<T>) stun_info_Ptr( dynamic_cast<StunInfo<T>*>( action_task_Ptr->get_action_info() ) );
        nassertv( stun_info_Ptr != NULL );
        PT(Object<T>) object_Ptr( action_task_Ptr->get_object() );
        nassertv( object_Ptr != NULL );
        
        object_Ptr->unstun();
    //}
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 StunInfo<T>::StunInfo( const double stun_duration )
             : ActionInfo<T>(),
               _stun_duration( stun_duration )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 StunInfo<T>::StunInfo( const StunInfo<T>& stun_info )
             : ActionInfo<T>( stun_info ),
               _stun_duration( stun_info._stun_duration )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 StunInfo<T>::~StunInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified StunInfo to this StunInfo.
  *
  * @param (const StunInfo<T>&) stun_info
  * @return StunInfo<T>&
  */
 template <typename T>
 inline StunInfo<T>& StunInfo<T>::operator=( const StunInfo<T>& stun_info )
 {
    ActionInfo<T>::operator=( stun_info );
    _stun_duration = stun_info._stun_duration;
    
    return *this;
 }
 
 /**
  * split_tasks() either assigns the specified task to the specified 
  * task manager or splits up the task into multiple tasks before 
  * assigning the tasks to the specified task manager.
  *
  * @param (ActionTask<T>*) action_task_Ptr
  * @param (AsyncTaskManager*) task_mgr_Ptr
  */
 template <typename T>
 inline void StunInfo<T>::split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr )
 {
    nassertv( action_task_Ptr != NULL );
    nassertv( task_mgr_Ptr != NULL );
    
    if ( _stun_duration != 0.0 )
    {
        // Stun the character
        PT(StunInfo<T>) stun_info_Ptr( dynamic_cast<StunInfo<T>*>(action_task_Ptr->get_action_info()) );
        PT(Object<T>) object_Ptr( action_task_Ptr->get_object() );
        nassertv( object_Ptr != NULL );
        object_Ptr->stun();
        
        // Create the ActionTask to be added to the AsyncTaskManager
        PT(ActionTask<T>) new_action_task_Ptr( new ActionTask<T>( object_Ptr, &StunInfo<T>::process_stun, stun_info_Ptr ) );
        new_action_task_Ptr->set_upon_death( &StunInfo<T>::upon_death_handler );
        
        task_mgr_Ptr->add( new_action_task_Ptr );
    }
 }
 
 /** STATIC FUNCTIONS **/
 
 #endif // STUN_INFO_H